<?php
/*
Plugin Name: WooCommerce Price Per Unit PRO
Plugin URI: https://mechcomp.cz/price-per-unit-pro/
Description: WooCommerce Price Per Unit allows the user to show prices recalculated per units(weight) and do some customization to the look of the prices
Version: 2.1.4
Author: Martin Mechura
Author URI: http://mechcomp.cz
Text Domain: woo-price-per-unit-pro
WC tested up to: 5.4.1
WC requires at least: 3.0

@author         Martin Mechura
@category    Admin

WooCommerce Price Per Unit. A Plugin that works with the WooCommerce plugin for WordPress.
Copyright (C) 2017 Martin Mechura
 */

namespace PPU;

if (!defined('ABSPATH')):
    exit; // Exit if accessed directly
endif;
define('MCMP_PPU_PLUGIN_VERSION','2.1.4');

class LicenceException extends \Exception {};
class GeneralException extends \Exception {};
class mcmp_PPU_pro
{
    // This is the secret key for API authentication. You configured it in the settings menu of the license manager plugin.
    private $SLM_SECRET_KEY = '5c9d3acf6951f9.25990198';
    // This is the URL where API query request will be sent to. This should be the URL of the site where you have installed the main license manager plugin. Get this value from the integration help page.
    private $SLM_LICENSE_SERVER_URL = 'https://mechcomp.cz/';
    // This is a value that will be recorded in the license manager data so you can identify licenses for this item/product.
    private $SLM_REFERENCE = 'WooCommerce Price Per Unit Pro';
    private static $instance = null;
    private $single_pr_id = 0;
    public $woo_version = '';
    public $dependency = null;
    public $hook_priority = 1010;
	public $plural_translations = array();
	//WooCommerce Product Table PRO
	public $wcpt_helper = null;
	//YITH WooCommerce Dynamic Pricing and Discounts Premium
	public $yith_dyn_pr_helper = null;
	// WPML Class container
	public $wpmlhelper = null;

    public static function get_instance()
    {
        if (null == self::$instance) {
            self::$instance = new self;
        }
        return self::$instance;
    }
    /**
     * The Constructor!
     * @since 1.0
     */
    public function __construct()
    {
        global $woocommerce;
        add_action( 'admin_init', array( $this, 'admin_activation_check' ) );
        if($this->dependency_checks()!==true){
            return;
        }
        //Loading translation
		add_action('init', array($this, 'plugin_init'));
		// Loading includes.
		$this->load_includes();
        //Action on plugin activation
		register_activation_hook(__FILE__, array($this, 'plugin_activated'));
		//Database upgrades.
		add_action( 'current_screen', array( $this, 'perform_db_upgrade' ) );
        //Loading style-sheet
        add_action('wp_enqueue_scripts', array($this, 'load_style'));
        add_action('admin_enqueue_scripts', array($this, 'load_admin_style'));

        //Adding general options tab
        add_filter('woocommerce_get_sections_products', array($this, 'add_general_options_section'));
        add_filter('woocommerce_get_settings_products', array($this, 'general_options'), 10, 2);

        //Improves cURL connection 
        add_filter('http_api_curl', array($this, 'change_curl_options'),10,3);
        if(!empty(get_option('mcmp_ppu_licence_key'))){
			//Load all helper classes.
			//add_action( 'plugins_loaded', array( $this, 'load_helpers' ), 9 );
			$this->load_helpers();
            $hook_priority = intval(get_option('_mcmp_ppu_hook_priority'));
            if(!empty($hook_priority)){
                $this->hook_priority = $hook_priority;
			}
            //Render the ppu field output on the front-end
			add_filter('woocommerce_get_price_html', array($this, 'custom_price'), $this->hook_priority, 2);
            add_filter('woocommerce_available_variation', array($this, 'custom_variant_price'), $this->hook_priority, 3);
			add_filter('woocommerce_cart_item_price', array($this, 'custom_cart_price'), $this->hook_priority, 3);
			//Custom filter to recalculated price
			add_filter('mcmp_get_recalc_price', array($this, 'custom_price'), $this->hook_priority, 2);
            //Displays additional info on product detail
            if(get_option('_mcmp_ppu_display_additional_info')=='yes'){
                add_filter( 'woocommerce_display_product_attributes', array($this, 'display_additional_info'),10,2);
                add_filter('woocommerce_available_variation', array($this, 'add_mcmp_unit_data'), 10, 3);
            }
            //Adding single product options tab
            add_filter('woocommerce_product_data_tabs', array($this, 'add_custom_product_options_tab'), 99, 1);
            //Adding single product options
            add_action('woocommerce_product_data_panels', array($this, 'product_options'));
            //Saving single product options
			add_action('woocommerce_process_product_meta', array($this, 'save_product_options'),30);
            //Adding variable product options
            add_action( 'woocommerce_product_after_variable_attributes', array($this,'product_variations_options'), 10, 3 );
            //Saving variable product options
			add_action( 'woocommerce_save_product_variation', array($this,'save_product_variations_options'), 10, 2 );
			
			if(get_option('_mcmp_ppu_orders_ppu_show')=='yes'){
				add_action( 'woocommerce_admin_order_item_headers', array( $this, 'ppu_add_order_column_header_item_ppu' ), 10, 1 );
				add_action( 'woocommerce_admin_order_item_values', array( $this, 'ppu_add_order_column_value_item_ppu' ), 10, 3 );
			}
			if(get_option('_mcmp_ppu_use_price_meta')=='yes'){
				//Adds meta containing price
				//On product save
				add_action('woocommerce_process_product_meta', array($this, 'mcmp_create_ppu_price_meta'),40);
				//On variation save
				add_action( 'woocommerce_save_product_variation', array($this,'mcmp_create_ppu_price_meta'), 20, 1 );
				//On quick edit save
				add_action( 'woocommerce_product_quick_edit_save', array($this,'mcmp_create_ppu_price_meta'), 10, 1 );
				//On bulk edit save
				add_action( 'woocommerce_product_bulk_edit_save', array($this,'mcmp_create_ppu_price_meta'), 10, 1 );
				
				//Syncs meta containing price with children
				//On product save
				add_action('woocommerce_process_product_meta', array($this, 'mcmp_sync_children_ppu_prices'),50);
				//On variation Ajax save
				add_action( 'woocommerce_ajax_save_product_variations', array($this,'mcmp_sync_children_ppu_prices'), 10, 1 );
				//On quick edit save
				add_action( 'woocommerce_product_quick_edit_save', array($this,'mcmp_sync_children_ppu_prices'), 20, 1 );
				//On bulk edit save
				add_action( 'woocommerce_product_bulk_edit_save', array($this,'mcmp_sync_children_ppu_prices'), 20, 1 );

				if(get_option('_mcmp_ppu_add_ppu_orderby')=='yes'){
					//Adds Order-by texts to the catalogue
					add_filter( 'woocommerce_catalog_orderby', array($this,'mcmp_add_custom_ordering'), 10, 1 );
					add_filter( 'woocommerce_default_catalog_orderby_options', array( $this, 'mcmp_add_custom_ordering' ), 10, 1 );
					//Handles Order-by
					add_filter( 'woocommerce_get_catalog_ordering_args', array( $this, 'mcmp_add_shop_ordering_args' ), 10, 3 );
				}

				//Display Price meta columns?
				if(get_option('_mcmp_ppu_display_product_cols')=='yes'){
					//Adds custom product cols
					add_filter( 'manage_edit-product_columns', array($this, 'mcmp_ppu_custom_product_columns')) ;
					//Connects custom product cols with data
					add_action( 'manage_product_posts_custom_column', array($this, 'mcmp_ppu_manage_product_columns'), 10, 2 );
				}
			}

			//helper for getting single product ID
            add_action('woocommerce_before_single_product', array($this, 'get_single_id'));
            add_action( 'woocommerce_admin_field_mcmp_ppu_button' , array($this,'admin_custom_button'));
            add_action( 'wp_ajax_run_recreate_prices', array($this, 'ajax_recreate_prices'));
        } else {
            add_action('after_plugin_row_' . plugin_basename(__FILE__),  array($this, 'plugin_extra_info'),10,3);
        }
        //Extending plugin actions
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'plugin_action_links'));
        //Hook woocommerce_admin_field_mcmplicence needs to be changed for new plugin - as well as 'type' in licence-settings.php
        add_action( 'woocommerce_admin_field_mcmplicence' , array($this,'admin_licence_field' ));
    }

	/**
     * Filters classes for individual PPU elements depending on used hook
     * @param string $classes - Existing classes
     * @return string 
     * @since 2.1.0
     */
	public function get_formatted_price($price){
		if(doing_filter('wcpt_element_markup')) {
			$price = wcpt_price( $price );
		}else{
			$price = wc_price( $price );
		}
		return $price;
	}


    public function add_mcmp_unit_data($data, $product, $variant){
        if(is_product()){
            $behav = $this->get_recalc_behaviour($product);
            if (!empty($behav)) {
                $plurals = false;
                $current_unit = false;
                $parent_product_id = $product->get_id();
                //Getting custom unit
                $ratio_unit = $this->get_option_override('_mcmp_ppu_custom_unit', $parent_product_id);
                if(empty($ratio_unit)){
                    //Getting different weight unit
                    $ratio_unit = $this->get_option_override('_mcmp_ppu_ratio_unit', $parent_product_id);
                    if(empty($ratio_unit)){
                        //Getting original shop weight
                        $ratio_unit = get_option('woocommerce_weight_unit');
                    }else{
                        $current_unit = get_option('woocommerce_weight_unit');
                    }
                }
                //Translation of the unit
				//First try to translate it through WPML, so if found it will override the translation from .po file.
				$trans_ratio_unit = apply_filters('mcmp_ppu_translate_wpml_reg_string', $ratio_unit, 'Price per Unit PRO', 'Predefined custom units names' );
                if(array_key_exists( $trans_ratio_unit, $this->plural_translations)){
                    $plurals = $this->plural_translations[ $trans_ratio_unit ];
                }else{
                    $trans_ratio_unit=esc_html_x( $trans_ratio_unit, 'Custom unit name', 'woo-price-per-unit-pro');
                }
                $weight_string = '';
                $var_id = $variant->get_id();
                $units = $this->get_option_override('_mcmp_ppu_cust_num_of_units', $var_id);
                if(empty($units)){
                    $units = $variant->get_weight();
                }
				$units = floatval( $units );
                if (!empty($units)) {
                    if(!empty($current_unit) && ($ratio_unit != $current_unit)){
                        $units = wc_get_weight($units, $ratio_unit, $current_unit);
                    }
                    if($plurals != false){
                        $trans_ratio_unit = esc_html(translate_nooped_plural( $plurals, $units, 'woo-price-per-unit-pro' ));
                    }
                    $weight_string = wc_format_localized_decimal($units) .' '.  $trans_ratio_unit;
                }

                if(!empty($weight_string)){
                    $data['mcmp_custom_unit'] = $units;
                    $data['mcmp_custom_unit_html'] = esc_html($weight_string);
                }
            }
        }
        return $data;
    }

    /**
     * Checks woocommerce version and remembers for later purpose
     * @param string $compare_version Version to check against, default is 3.0
     * @return boolean is the WooCommerce version greater than $compare_version
     * @since 1.5
     */
    public function check_woo_version($compare_version = '3.0',$compare_operator = '>=')
    {
        $version = $this->woo_version;
        if (empty($version)) {
            $version=get_option( 'woocommerce_version' );
            $this->woo_version = $version;
        }
        if (version_compare($version, $compare_version, $compare_operator)) {
            return true;
        }
        return false;
    }

    /**
     * Displays admin notice - when in WooCommerce 4 it handles the display through transients and function "display_stored_messages"
     * @param string $message Message to output
     * @param string $type Type of the message
     * @param string $transient Name of transient to store the message
     * @since 1.8
     */
    public function mcmp_add_message($message, $type, $transient = '')
    {
        if(empty($message) or !is_string($message)){
            return;
        }
        $class='notice is-dismissible ';
        switch ($type) {
            case 'success':
            case 'warning':
            case 'error':
            case 'info':
                $class.='notice-'.$type;
            break;
            default:
                $class.='notice-info';
            break;
        }
        $output = '<div class="' . $class . '"><p>' . wp_kses_post($message) . '</p></div>';
        if (!empty($transient)){
            $notice_num = 0;
            $trans_num = $transient . $notice_num;
            while (get_transient($trans_num)!=false){
                $trans_num = $transient . ++$notice_num;
            } 
            set_transient( $trans_num, $output, 60 );
        }else{
            echo $output;
        }
    }

	/**
	 * Function used to Load all includes.
	 *
	 * @since 2.1.0
	 */
	public function load_includes() {
		include_once plugin_dir_path( __FILE__ ) . 'includes/public-functions.php';
		include_once plugin_dir_path( __FILE__ ) . 'includes/plugin-checkers.php';
	}

	/**
	 * Function used to Load all hepler classes.
	 *
	 * @since 2.1.1
	 */
	public function load_helpers() {
		if(\mcmp_is_wcpt_active()){
			$this->wcpt_helper = include_once plugin_dir_path( __FILE__ ) . 'helpers/class-wcpt_helper.php';
		}
		if(\mcmp_is_yith_dyn_pr_active()){
			$this->yith_dyn_pr_helper = include_once plugin_dir_path( __FILE__ ) . 'helpers/class-yith_dyn_pr_helper.php';
		}
		if(\mcmp_is_wpml_active()){
			$this->wpmlhelper = include_once plugin_dir_path( __FILE__ ) . 'helpers/class-wpmlhelper.php';
		}
	}

    /**
     * Displays modified units info on product detail
     * @param array $product_attributes - attributes of the product to display
     * @param object $product - affected product
     * @return array altered array of attributes
     * @since 2.0.6
     */
    public function display_additional_info($product_attributes, $product)
    {
        $behav = $this->get_recalc_behaviour($product);
        if (!empty($behav)) {
            $plurals = false;
            $replace_weight = false;
            $parent_product_id = $this->check_product_id($product, true);
            $parent_prod_weight = $product->get_weight();
            $add_text = $this->get_option_override('_mcmp_ppu_recalc_text', $parent_product_id);
            $ratio_unit = $this->get_option_override('_mcmp_ppu_custom_unit', $parent_product_id);
            if(empty($ratio_unit) || $add_text != '-automatic-'){
                $ratio_unit = $this->get_option_override('_mcmp_ppu_ratio_unit', $parent_product_id);
                if(empty($ratio_unit)){
                    $ratio_unit = get_option('woocommerce_weight_unit');
                    $unit_label = __( 'Net weight', 'woo-price-per-unit-pro' );
                }else{
                    //Take original WooCommerce weight label so I don't have translation in my text domain
                    $current_unit = get_option('woocommerce_weight_unit');
                    if($ratio_unit == $current_unit){
                        $unit_label = __( 'Net weight', 'woo-price-per-unit-pro' );
                    }else{
                        $unit_label = $product_attributes['weight']['label']; 
                        $replace_weight = true;
                    }
                }
            }else{
                $unit_label = __( 'Number of %s', 'woo-price-per-unit-pro' );
            }
			//First try to translate it through WPML, so if found it will override the translation from .po file.
			$trans_ratio_unit = apply_filters('mcmp_ppu_translate_wpml_reg_string', $ratio_unit, 'Price per Unit PRO', 'Predefined custom units names' );
            if(array_key_exists( $trans_ratio_unit, $this->plural_translations)){
                $plurals = $this->plural_translations[ $trans_ratio_unit ];
            }else{
                $trans_ratio_unit=esc_html_x($trans_ratio_unit, 'Custom unit name', 'woo-price-per-unit-pro');
            }
            switch ($product->get_type()) {
                case 'variable':
                    $custom_units_string = '';
                    //When getting variants the filter will fire this again - this is to speed up the process
                    remove_filter('woocommerce_get_price_html', array($this, 'custom_price'), $this->hook_priority);
                    $variations = $product->get_available_variations();
                    add_filter('woocommerce_get_price_html', array($this, 'custom_price'), $this->hook_priority, 2);
                    $num_of_variants = count($variations);
                    if ($num_of_variants > 0) {
                        foreach($variations as $value){
                            $var_id = $value['variation_id'];
                            $units = $this->get_option_override('_mcmp_ppu_cust_num_of_units', $var_id);
                            if($replace_weight == true){
                                if(empty($units)){
                                    $units=!empty($value['weight']) ? $value['weight'] : $parent_prod_weight;
                                }
                                $units = wc_get_weight($units, $ratio_unit, $current_unit);
                            }
							$units = floatval( $units );
                            if (!empty($units)) {
                                if($plurals != false){
                                    $trans_ratio_unit = esc_html(translate_nooped_plural( $plurals, $units, 'woo-price-per-unit-pro' ));
                                }
                                $custom_units_string .= (empty($custom_units_string) ? '':', ') . wc_format_localized_decimal($units) .' '.  $trans_ratio_unit;
                            }
                        }
                    }
                break;

                case 'simple':
                    $units = $this->get_option_override('_mcmp_ppu_cust_num_of_units', $parent_product_id);
                    if($replace_weight == true){
                        if(empty($units)){
                            $units = $parent_prod_weight;
                        }
                        $units = wc_get_weight($units, $ratio_unit, $current_unit);
                    }
					$units = floatval( $units );
                    if (!empty($units)) {
                        if($plurals != false){
                            $trans_ratio_unit = esc_html(translate_nooped_plural( $plurals, $units, 'woo-price-per-unit-pro' ));
                        }
                        $custom_units_string = wc_format_localized_decimal($units) .' '.  $trans_ratio_unit;
                    }
                break;
            }
            if(!empty($custom_units_string)){
                if($replace_weight == true){
                    unset($product_attributes['weight']);
                }
                if($plurals != false){
                    $trans_ratio_unit = esc_html(translate_nooped_plural( $plurals, 0, 'woo-price-per-unit-pro' ));
                }
                $my_attributes['mcmp_custom_unit'] = array(
                    'label' => sprintf($unit_label, $trans_ratio_unit),
                    'value' => $custom_units_string,
                );
                $product_attributes = array_merge($my_attributes,$product_attributes);
            }
        }   
        return $product_attributes;
    }

    /**
     * Displays message stored in transient
     * @param string $message Message to output
     * @param string $type Type of the message
     * @since 2.0.1
     */
    public function display_stored_messages($transient = 'ppu-notice')
    {
        $notice_num = 0;
        $trans_num = $transient . $notice_num;
        $message = get_transient($trans_num);
        while ($message!=false){
            delete_transient($trans_num);
            echo $message;
            $trans_num = $transient . ++$notice_num;
            $message = get_transient($trans_num);
        } 
    }

    /**
     * Changes timeout for cUrl
     * @since 1.9.5
     */
    public function change_curl_options($handle, $r, $url)
    {
        curl_setopt($handle, CURLOPT_CONNECTTIMEOUT_MS, 30000);
        return $handle;
    }

    /**
     * Truncates number to $precision decimal points
     * @param float $number Number to truncate
     * @param float $precision Number of digits after decimal point
     * @return float trimmed number
     * @since 1.9.5
     */
    public function truncate_number($number, $precision = 0)
    {
        if(function_exists('bcadd')){
            return floatval(bcadd($number, 0, $precision));
        }else{
            return floatval(preg_replace('/\.(\d{'.intval($precision).'}).*/', '.$1', $number));
        }
    }

    /**
     * Displays licence field in WooCommerce admin
     * @param array $value Information about the field
     * @since 1.9.5
     */
    public function admin_licence_field($value)
    {
        include 'includes/licence-field-display.php';
	}
	
	/**
     * Adds custom fields to product ordering
     * @param array $fields - Custom fields for ordering
     * @since 2.0.8
     */
	public function mcmp_add_custom_ordering($fields)
	{
		$fields['ppu_price-asc'] 	 = __( 'Sort by Price per unit: low to high', 'woo-price-per-unit-pro' );
		$fields['ppu_price-desc'] = __( 'Sort by Price per unit: high to low', 'woo-price-per-unit-pro' );
		return $fields;
	}
	
	/**
	 * Handles Order-by of products
	 * @param array $args - Order-by arguments.
	 * @param string $orderby - Order-by field.
	 * @param string $order - Order-by direction.
	 * @return array
	 * @since 2.0.8
	 */
	public function mcmp_add_shop_ordering_args($args, $orderby, $order)
	{
		switch ( $orderby ) {
			case 'ppu_price':
				$order = $order == 'DESC'? 'DESC' : 'ASC';
				$args['orderby']  = array(
					'meta_value_num' => $order,
					'price' => $order,
				);
				if ( $order== 'ASC' ){
					add_filter( 'woocommerce_product_query_meta_query', array( $this, 'mcmp_add_meta_query_ppu_asc' ), 10, 2 );
				}else{
					add_filter( 'woocommerce_product_query_meta_query', array( $this, 'mcmp_add_meta_query_ppu_desc' ), 10, 2 );
				}
			break;
		}
		return $args;
	}

	/**
	 * Modifies the query for custom ordering. This is a workaround,
	 * because when I use 'meta_key' in ordering, products without the meta are excluded from the result.
	 * @param array $meta_query - Meta query construct.
	 * @param string $query - Actual query.
	 * @return array
	 * @since 2.0.9
	 */
	public function mcmp_add_meta_query_ppu_asc($meta_query, $query)
	{
		$meta_query = array(
			'relation' => 'OR',
			array(
				'key' => '_mcmp_ppu_price_min', 
				'compare' => 'NOT EXISTS'
			),
			array(
				'key' => '_mcmp_ppu_price_min', 
				'compare' => 'EXISTS'
			)
		);
		return $meta_query;
	}

	/**
	 * Modifies the query for custom ordering. This is a workaround,
	 * because when I use 'meta_key' in ordering, products without the meta are excluded from the result.
	 * @param array $meta_query - Meta query construct.
	 * @param string $query - Actual query.
	 * @return array
	 * @since 2.0.9
	 */
	public function mcmp_add_meta_query_ppu_desc($meta_query, $query)
	{
		$meta_query = array(
			'relation' => 'OR',
			array(
				'key' => '_mcmp_ppu_price_max', 
				'compare' => 'NOT EXISTS'
			),
			array(
				'key' => '_mcmp_ppu_price_max', 
				'compare' => 'EXISTS'
			)
		);
		return $meta_query;
	}

    /**
     * Displays a custom button in WooCommerce admin
     * @param array $value Information about the field
     * @since 2.0.8
     */
    public function admin_custom_button($value)
    {
        include 'includes/admin-custom-button-display.php';
    }

    /**
     * Runs AJAX on script
     * @since 2.0.8
     */
    public function ajax_recreate_prices()
    {
        include 'includes/ajax-recreate-prices.php';
    }

    /**
     * Checks if licence is valid
     * @since 1.8
     */
    public function license_management($licence)
    {   
        /*** License activate button was clicked ***/
		global $wp_version;
        if (isset($_REQUEST["activate${licence}"]) || isset($_REQUEST["deactivate${licence}"])) {
			$license_key = filter_var($_REQUEST["${licence}_to_check"], FILTER_SANITIZE_STRING);
			try{
				if(empty($license_key)){
					$message=esc_html__('Licence key cannot be empty.','woo-price-per-unit-pro');
					throw new GeneralException( $message );
				}
				$activate = isset($_REQUEST["activate${licence}"]) ? true : false;
				$url = home_url(); 
				$domain =  parse_url($url,PHP_URL_HOST) . parse_url($url,PHP_URL_PATH);
				if(is_multisite()){
					$domain .= '-[' . get_current_blog_id() . ']';
				}
				//Error message
				if($activate === true){
					$error	 = esc_html__('There was an error when activating licence!','woo-price-per-unit-pro');
					$success = esc_html__('Licence activation was successful.','woo-price-per-unit-pro');
				}else{
					$error	 = esc_html__('There was an error when de-activating licence!','woo-price-per-unit-pro');
					$success = esc_html__('Licence de-activation was successful.','woo-price-per-unit-pro');
				}

				// Http query params
				$http_args = array(
                    'timeout' => 20,
                    'user-agent' => 'WordPress/' . $wp_version . '; ' . home_url('/'),
                    'sslverify' => true,
				);

				//Check the licence first
				if($activate){
					// API query parameters - Check
					$api_params = array(
						'slm_action' => 'slm_check',
						'secret_key' => $this->SLM_SECRET_KEY,
						'license_key' => $license_key,
						'query_domain' => $domain,
						'time' => time(),
					);
					// Send query to the license manager server
					$query = esc_url_raw(add_query_arg($api_params, $this->SLM_LICENSE_SERVER_URL));
					$response = wp_remote_get($query, $http_args);
					
					// Check for error in the response
					if (is_wp_error($response)){
						$message = esc_html__('Unexpected Error! The query returned with an error:','woo-price-per-unit-pro') . esc_html($response->get_error_message());
						throw new GeneralException( $message );
					}
	
					// Decoding response
					$license_data = json_decode(wp_remote_retrieve_body($response), true);
	
					//No response at all
					if (!$license_data){
						$message = esc_html__('Licencing server is experiencing problems, or is not available. ','woo-price-per-unit-pro');
						throw new GeneralException( $message );
					}

					if ('error' == $license_data['result']){
						$message = esc_html__('Message from the server: ','woo-price-per-unit-pro') . esc_html($license_data['message']);
						throw new LicenceException( $message );
					}

					if(array_key_exists('status',$license_data)){
						if('blocked'===$license_data['status']){
							$message = esc_html__('This licence was blocked.','woo-price-per-unit-pro');
							throw new LicenceException( $message );
						}
						if('expired'===$license_data['status']){
							$message = esc_html__('This licence has expired.','woo-price-per-unit-pro');
							throw new LicenceException( $message );
						}
					}

					//Already registered
					if(array_key_exists('registered_domains',$license_data)){
						$reg_doms = $license_data['registered_domains'];
						if( in_array( $domain, array_column($reg_doms, 'registered_domain'),true)){
							update_option("${licence}_key", $license_key); 
							$success .= '</br>'.esc_html__('Licence was already registered for this domain.','woo-price-per-unit-pro');
							$this->mcmp_add_message($success,'sucess','ppu-licence-notice');
							//Save the license key in the options table
							return;
						}
						if( count($reg_doms) >= intval($license_data['max_allowed_domains'])){
							$message = esc_html__('Maximum allowed domains already registered for this licence.','woo-price-per-unit-pro');
							throw new LicenceException( $message );
						}
					}
				}

				// API query parameters - Activate/Deactivate
                $api_params = array(
                    'slm_action' => $activate === true ? 'slm_activate':'slm_deactivate',
                    'secret_key' => $this->SLM_SECRET_KEY,
                    'license_key' => $license_key,
                    'registered_domain' => $domain,
					'item_reference' => urlencode($this->SLM_REFERENCE),
					'time' => time(),
                );
				// Send query to the license manager server
				$query = esc_url_raw(add_query_arg($api_params, $this->SLM_LICENSE_SERVER_URL));
				$response = wp_remote_get($query, $http_args);

				// Check for error in the response
				if (is_wp_error($response)){
					$message = esc_html__('Unexpected Error! The query returned with an error:','woo-price-per-unit-pro') . esc_html($response->get_error_message());
					throw new GeneralException( $message );
				}

				// Decoding response
				$license_data = json_decode(wp_remote_retrieve_body($response), true);

				//No response at all
				if (!$license_data){
					$message = esc_html__('Licencing server is experiencing problems, or is not available. ','woo-price-per-unit-pro');
					throw new GeneralException( $message );
				}

				if ( 'error' == $license_data['result'] ){
					$message = esc_html__('Message from the server: ','woo-price-per-unit-pro') . esc_html($license_data['message']);
					throw new LicenceException( $message );
				}

				if ( 'success' == $license_data['result'] ){
					if($activate){
						update_option("${licence}_key", $license_key);
					}else{
						update_option("${licence}_key", '');
					}
					$success .='</br>' . esc_html__('Message from the server: ','woo-price-per-unit-pro') . esc_html($license_data['message']);
                    $this->mcmp_add_message($success,'sucess','ppu-licence-notice');
				}
			} 
			catch ( GeneralException $e ) {
				if ( $e->getMessage() ) {
					$error .= '</br>' . $e->getMessage();
				}
				$this->mcmp_add_message($error,'error','ppu-licence-notice');
			}
			catch ( LicenceException $e ) {
				if ( $e->getMessage() ) {
					$error .= '</br>' .$e->getMessage();
				}
				$this->mcmp_add_message($error,'error','ppu-licence-notice');
				//Clear the license
				update_option("${licence}_key", ''); 
			}
        }
    }        

    /**
     * Load plugin's text-domain
     * @since 1.0
     */
    public function plugin_init()
    {
        load_plugin_textdomain('woo-price-per-unit-pro', false, dirname(plugin_basename(__FILE__)) . '/languages/');
		include_once plugin_dir_path( __FILE__ ) . 'includes/dummy-translations.php';
    }

    /**
     * Checks if requirements for activation are met
     * Now it tests if free version is active
     * @return mixed String with error message or true when dependency checks are all-right
     * @since 1.9
     */
    public function dependency_checks()
    {
        if (!is_null($this->dependency)){
            return $this->dependency;
        }
        if ( in_array( 'woo-price-per-unit/woo-price-per-unit.php', (array) get_option( 'active_plugins' ), true ) 
            || (is_multisite() && array_key_exists( 'woo-price-per-unit/woo-price-per-unit.php', (array) get_site_option( 'active_sitewide_plugins' ) ) ) ) {
                $this->dependency =  esc_html__('Price per unit plugin - free version is active. Please deactivate it first.', 'woo-price-per-unit-pro');
                return $this->dependency;        
        }
        $this->dependency = true;
        return $this->dependency;
    }

    /**
     * Prevents activation of the plugin when dependency checks fail
     * @since 1.9
     */
    function admin_activation_check()
    {
        $dependency_info=$this->dependency_checks();
        if($dependency_info!==true){            
            if ( is_plugin_active( plugin_basename( __FILE__ ) ) ) {
                deactivate_plugins( plugin_basename( __FILE__ ) );
                $type='error';
                //Passing arguments into add_action by using PHP lambda - nice workaround
                add_action('admin_notices', function() use ( $dependency_info,$type) { 
                    $this->mcmp_add_message($dependency_info,$type); });
                if ( isset( $_GET['activate'] ) ) {
                    unset( $_GET['activate'] );
                }
            }
        }
    }

    /**
     * Action on plugin activation - currently setting defaults
     * Checks also the conditions for plugin activation - if not it prevents the activation
     * @since 1.5
     */
    public function plugin_activated()
    {
        $dependency_info=$this->dependency_checks();
        if($dependency_info!==true){
            deactivate_plugins( plugin_basename( __FILE__ ) );
            wp_die(esc_html__('Plugin NOT activated: ', 'woo-price-per-unit-pro') . $dependency_info);
        }
        include 'includes/general-setting-defaults.php';
	}
	
	/**
	 * Performs upgrade on plugin update
	 *
	 * @since 2.1.0
	 */
	public function perform_db_upgrade() {
		$used_db_version = get_option( '_mcmp_ppu_db_version', '1.0.0' );
		if ( MCMP_PPU_PLUGIN_VERSION != $used_db_version ){
			include 'includes/database-upgrade.php';
		}
	}

    /**
     * Legacy options removal and migrating to new settings
     * @since 1.6
     */
    public function legacy_options_fix()
    {
        $old_opt = get_option('_mcmp_ppu_general_behaviour');
        if ($old_opt !== false) {
            $old_opt = (empty($old_opt)) ? 'not' : $old_opt;
            update_option('_mcmp_ppu_general', $old_opt);
            delete_option('_mcmp_ppu_general_behaviour');
        }
        $old_opt = get_option('_mcmp_ppu_single_page_behaviour');
        if ($old_opt !== false) {
            $old_opt = (empty($old_opt)) ? 'not' : $old_opt;
            update_option('_mcmp_ppu_single_page', $old_opt);
            delete_option('_mcmp_ppu_single_page_behaviour');
        }
        $old_opt = get_option('_mcmp_ppu_cart_page');
        switch ($old_opt) {
            case 'single':
                $old_opt = '_mcmp_ppu_single_page-add';
                update_option('_mcmp_ppu_cart_page', $old_opt);
            break;
            case 'shop':
                $old_opt = '_mcmp_ppu_general-add';
                update_option('_mcmp_ppu_cart_page', $old_opt);
            break;
        }
    }

    /**
     * Load plugin's CSS style
     * @since 1.3
     */
    public function load_style()
    {
		if(get_option('_mcmp_ppu_display_additional_info')=='yes'){
			wp_register_script('woo-ppu-front-variable', plugins_url('/assets/JS/woo-ppu-front-variable.js', __FILE__), array('jquery','wc-add-to-cart-variation'), MCMP_PPU_PLUGIN_VERSION);
			wp_enqueue_script('woo-ppu-front-variable');
		}
		wp_enqueue_style('mcmp_price_per_unit_style', plugins_url('/assets/CSS/woo-ppu-default-style.css', __FILE__), array(), MCMP_PPU_PLUGIN_VERSION );
        if (get_option('_mcmp_ppu_add_row_css') == 'yes') {
            wp_enqueue_style('mcmp_price_per_altered_style', plugins_url('/assets/CSS/woo-ppu.css', __FILE__), array(), MCMP_PPU_PLUGIN_VERSION );
		}
		if(\mcmp_is_wcpt_active()){
			wp_enqueue_style('mcmp_wcpt_style', plugins_url('/assets/CSS/woo-ppu-wcpt-style.css', __FILE__), array(), MCMP_PPU_PLUGIN_VERSION );
		}
    }

    /**
     * Load plugin's admin styles and scripts
     * @since 1.6
     */
    public function load_admin_style($hook)
    {
        wp_register_script('mcmp-ppu-admin', plugins_url('/assets/JS/woo-ppu-admin.js', __FILE__), array('jquery'), MCMP_PPU_PLUGIN_VERSION );
        wp_enqueue_script('mcmp-ppu-admin');
        if($hook == 'post.php'){
            wp_register_script('woo-ppu-admin-variable', plugins_url('/assets/JS/woo-ppu-admin-variable.js', __FILE__), array('jquery','wc-admin-meta-boxes'), MCMP_PPU_PLUGIN_VERSION );
            wp_enqueue_script('woo-ppu-admin-variable');
        }
        wp_register_script('woo-ppu-admin-ajax_scripts', plugins_url('/assets/JS/woo-ppu-admin-ajax_scripts.js', __FILE__), array('jquery'), MCMP_PPU_PLUGIN_VERSION );
        wp_enqueue_script('woo-ppu-admin-ajax_scripts');
        wp_localize_script( 'woo-ppu-admin-ajax_scripts', 'ajax_object',
            array( 'ajax_url' => admin_url( 'admin-ajax.php' )) );

        wp_enqueue_style('mcmp_price_per_unit_admin_style', plugins_url('/assets/CSS/woo-ppu-admin.css', __FILE__), array(), MCMP_PPU_PLUGIN_VERSION );
        if(empty(get_option('mcmp_ppu_licence_key'))){
            wp_enqueue_style('mcmp_price_per_unit_plugin_admin_style', plugins_url('/assets/CSS/woo-ppu-plugin-admin.css', __FILE__), array(), MCMP_PPU_PLUGIN_VERSION );
        }
    }

    /**
     * Add settings link
     * @return string HTML output into plugin info
     * @since 1.0
     */
    public function plugin_action_links($links)
    {
        $plugin_links = array(
            '<a href="' . admin_url('admin.php?page=wc-settings&tab=products&section=mcmp_price_pu') . '">' . esc_html__('Settings & Licence', 'woo-price-per-unit-pro') . '</a>',
        );
        return array_merge($plugin_links, $links);
    }

    /**
     * Adds a row after plugin info - used for activate plugin button
     * @param string $plugin_file
     * @param array $plugin_data
     * @param string $status
     * @since 1.9.6
     */
    public function plugin_extra_info($plugin_file, $plugin_data, $status)
    {   
        ?>  
            <tr class="active mcmp-plugin-extra-info ">
                <th class="check-column"></th>
                <td colspan="2">
                    <?php echo esc_html__('The plugin licence is not activated', 'woo-price-per-unit-pro')?>
                    <a class="button mcmp_activate" href= <?php echo '"' . admin_url('admin.php?page=wc-settings&tab=products&section=mcmp_price_pu').'">' . esc_html__('Activate licence', 'woo-price-per-unit-pro')?> </a>
               </td>
            </tr>
        <?php
    }
    
    /**
     * Adds tab to product options
     * @param array $product_opt_tabs Newly created tab
     * @since 1.0
     */
    public function add_custom_product_options_tab($product_opt_tabs)
    {
        $product_opt_tabs['price-per-unit'] = array(
            'label' => esc_html__('Price Per Unit PRO', 'woo-price-per-unit-pro'), // translatable
            'target' => 'mcmp_ppu_options', // translatable
        );
        return $product_opt_tabs;
    }

    /**
     * @param array $atts Outputs options to show on plugin general settings tab
     * Add the custom fields to the product general tab
     * @since 1.0
     */
    public function product_options($atts)
    {
        global $woocommerce, $post;
        include 'includes/product-settings.php';
    }
    
    /**
     * Update the database with the new options
     * @param integer $post_id Id of the post where setting will be saved
     * @since 1.0
     */
    public function save_product_options($post_id)
    {
        include 'includes/product-settings-save.php';
    }

    /**
     * Add the custom fields to the product variations
     * @param string $loop Id of the pass of variant
     * @param array $variation_data Passed information about the variant
     * @param object $variation Object containing the variation
     * @since 1.7
     */
    public function product_variations_options($loop, $variation_data, $variation)
    {
        include 'includes/variation-settings.php';
    }

    /**
     * Update the database with the new variations options
     * @param integer $variation_id ID of variation for which we want to save data
     * @param string $loop Id of the pass of variant
     * @since 1.7
     */
    public function save_product_variations_options($variation_id, $loop)
    {
        include 'includes/variation-settings-save.php';
	}

    /**
     * Syncs recalc price with it's children - should be called after the product recalc prices are calculated
     * @param integer $product_id - ID of product for which we want to sync price with children.
     * @since 2.0.8
     */

    public function mcmp_sync_children_ppu_prices($product_to_inspect)
    {
		if (is_numeric($product_to_inspect)) {
			$product = wc_get_product($product_to_inspect);
			$product_id = $product_to_inspect;
		}
		if (is_object($product_to_inspect)) {
			$product = $product_to_inspect;
			$product_id = $product_to_inspect->get_id();
		}
		if($product->get_type() ==='variable'){
			$children = $product->get_children();
		}else{
			$children = array($product_id);
		}
		$prices = array();
		foreach ($children as $child_id) {
			$price = get_post_meta($child_id,'_mcmp_ppu_price',true);
			if(!empty($price)) {
				$prices[] = $price;					
			}
		}    
		if ($prices) {
			asort($prices);
			$price_min = reset($prices);
			$price_max = end($prices);
			update_post_meta( $product_id, '_mcmp_ppu_price_min', $price_min );
			update_post_meta( $product_id, '_mcmp_ppu_price_max', $price_max );
		}else{
			delete_post_meta( $product_id, '_mcmp_ppu_price_min' );
			delete_post_meta( $product_id, '_mcmp_ppu_price_max' );
		}
	}

	/**
     * Creates meta data for the product - should be called after the product recalc prices are calculated
     * @param mixed $product_to_inspect - product for which we want to create the meta
     * @since 2.0.8
     */

	public function mcmp_create_ppu_price_meta($product_to_inspect)
    {
		if (is_numeric($product_to_inspect)) {
			$product = wc_get_product($product_to_inspect);
			$product_id = $product_to_inspect;
		}
		if (is_object($product_to_inspect)) {
			$product = $product_to_inspect;
			$product_id = $product_to_inspect->get_id();
		}
		if($product->get_type() ==='variable'){
			$children = $product->get_children();
		}else{
			$children = array($product_id);
		}
		if($this->should_recalc($product_id, true)){
			foreach ($children as $value) {
				$price = $this->calculate_product_price($value, true);
				if(!empty($price)){
					$text_price = $this->mcmp_format_plain_price($price);
					$text_price = $this->render_recalc_text($value, $text_price, false, false);
					update_post_meta($value ,'_mcmp_ppu_price', $price);
					update_post_meta($value ,'_mcmp_ppu_price_text', $text_price);
				}else{
					delete_post_meta( $value, '_mcmp_ppu_price');
					delete_post_meta( $value, '_mcmp_ppu_price_text');
				}
			}
		}else{
			foreach ($children as $value) {
				delete_post_meta( $value, '_mcmp_ppu_price');
				delete_post_meta( $value, '_mcmp_ppu_price_text');
			}
		}
	}	

	/**
	 * Modification of order details display - Header order item - Price per Unit
	 *
	 * @since 2.0.9
	 * @param object $order - Modified order.
	 */
	public function ppu_add_order_column_header_item_ppu( $order ) {
		$column_name = esc_html__( 'Price per unit', 'woo-sell-by-weight-pro' );
		echo '<th class="item_ppu_price sortable" data-sort="float">' . $column_name . '</th>';
	}

	/**
	 * Modification of order details display - Column order item - Price per Unit
	 *
	 * @since 2.0.9
	 * @param object $product - Modified order item product.
	 * @param object $item - Modified order item.
	 * @param object $item_id - Modified order item ID.
	 */
	public function ppu_add_order_column_value_item_ppu( $product, $item, $item_id = null ) {
		if ( 'line_item' === $item->get_type() ) {
			$prod_id = $product->get_id();
			$output = '<td></td>';
			if($this->should_recalc($prod_id, true)){
				$value = $this->calculate_product_price($prod_id, true);
				if ( ! empty( $value ) ) {
					$form_value = $this->mcmp_format_plain_price($value);
					$form_value = $this->render_recalc_text($prod_id, $form_value, false, false);
					$output = '<td class="item_ppu_price" data-sort-value="' . $value . '">' . $form_value . '</td>';
				}
			}
			echo $output;
		}
	}
	
    /**
     * Add Price per Unit settings section under the Products tab.
     * @param array $sections List of sections
     * @since 1.0
     */
    public function add_general_options_section($sections)
    {
        $sections['mcmp_price_pu'] = esc_html__('Price Per Unit PRO', 'woo-price-per-unit-pro');
        return $sections;
    }

    /**
     * Populates Price per Unit settings section under the Products tab.
     * @param array $settings Settings of WooCommerce
     * @param string $current_section Which section is populated right now
     * @since 1.0
     */
    public function general_options($settings, $current_section)
    {
        if ($current_section == 'mcmp_price_pu') {
            $this->license_management('mcmp_ppu_licence');
            $this->display_stored_messages('ppu-licence-notice');
            $cust_settings = include 'includes/licence-settings.php';
            if(!empty(get_option('mcmp_ppu_licence_key'))){
                $gen_settings = include 'includes/general-settings.php';
                $cust_settings = array_merge($cust_settings,$gen_settings);
            }
            return $cust_settings;
            // If not, return the standard settings
        } else {
            return $settings;
        }
    }

    /**
     * Saves product id from single product view
     * needed for determining if not running from widget on single page
     * @since 1.0
     */
    public function get_single_id()
    {
        global $product;
        if(is_object($product)){
            $this->single_pr_id = $product->get_id();
        }
    }

     /**
     * Checks if the product is valid and then returns it's ID
     * @since 2.0
     * @param mixed $product_to_inspect which product to check
     * @param boolean $get_parent whether to get parent product ID
     * @return mixed ID of the verified product of false when wrong Product
     */
    public function check_product_id($product_to_inspect, $get_parent = false)
    {
        $product_id = false;
        if (!empty($product_to_inspect)) {
            if (is_numeric($product_to_inspect)) {
                $product_to_inspect = wc_get_product($product_to_inspect);
            }
            if (is_object($product_to_inspect)) {
                $product_id = $product_to_inspect->get_id();
                if($get_parent == true && $product_to_inspect->get_type() === 'variation'){
                    $product_id =  $product_to_inspect->get_parent_id();
                }
            }
        }
        return $product_id;
    }

    /**
     * Gets option with product option overrides
     * Works with product meta to check for option, if value is empty proceed to general options
     * @since 1.6
     * @param string $option which option to check
     * @param integer $product_id_to_inspect which product to check - supposed to be checked first
     * @param mixed $default value which should be passed if nothing is found
     * @return string
     */
    public function get_option_override($option, $product_id_to_inspect, $default = '')
    {
        $product_override = $option . '_override';
        $option_val = '';
        //Getting parent product ID of $product_id_to_inspect
        $product_id = $this->check_product_id($product_id_to_inspect, true);
        //When the ID differs it means that we deal with variant and that needs extra options
        //Variant options
        if($product_id != $product_id_to_inspect){
            $option_val = get_post_meta($product_id_to_inspect, $product_override, true);
        }
        //Product options
        if (empty($option_val) && is_numeric($product_id)) {
            $option_val = get_post_meta($product_id, $product_override, true);
        }
        //General options
        if (empty($option_val)) {
            $option_val = get_option($option);
        }
        //Everything failed and we have default? Place it there. The ifs order is this because default is rarely used 
        if (!empty($default)) {
            if (empty($option_val)) {
                $option_val = $default;
            }
        }
        return $option_val;
    }

    /**
     * Gets behaviour of recalculation
     * The product is checked - on problems returns false
     * @since 1.6
     * @param mixed $product_to_inspect product id of product object
     * @param string $special_case Alter the behaviour on special occasion
     * @return mixed false if should not recalculate or string type of behaviour for recalculation
     *
     */
    public function get_recalc_behaviour($product_to_inspect, $special_case = false)
    {
        $product_id = $this->check_product_id($product_to_inspect, true);
        if (empty($product_id)) {
            return false;
        }
        switch ($special_case) {
            case 'cart':
                $option = get_option('_mcmp_ppu_cart_page');
                $delim_pos = strpos($option,'-');
                if($delim_pos == false){
                    $behav = $option;
                }else{
                    $behav = substr($option,$delim_pos+1);
                    $option = substr($option,0,$delim_pos);
                    $behav = ($this->get_option_override($option, $product_id) === 'not') ? false : $behav;
                }
            break;

            default:
                //Determine whether to recalculate or not - depending also on override
                if (is_product() && $product_id === $this->single_pr_id) {
                    //Single product page + is it That product or some widget product?
                    $option = '_mcmp_ppu_single_page';
                } else {
                    //Other pages
                    $option = '_mcmp_ppu_general';
                }
                $behav = $this->get_option_override($option, $product_id);
            break;
        }
        $behav = ($behav === 'not') ? false : $behav;
        return $behav;
    }

    /**
     * Queries if recalculation is set to happen for the product
     * @since 2.0.8
     * @param mixed $product_to_inspect product id of product object
     * @param boolean $bypass_check bypass product checking - supposed to be checked - $product_to_inspect must be id of the product
     * @return boolean recalc happens or not
     *
     */
    public function should_recalc($product_to_inspect, $bypass_check = false)
    {
        if($bypass_check === true){
            $product_id = $product_to_inspect;
        }else{
            $product_id = $this->check_product_id($product_to_inspect, true);
        }
        if (empty($product_id)) {
            return false;
        }
        $product = wc_get_product($product_id);
        $prod_type = $product->get_type();
        if($prod_type==='simple' || $prod_type ==='variable' || $prod_type ==='variation'){
            $option = '_mcmp_ppu_single_page';
            $behav = $this->get_option_override($option, $product_id);
            if($behav ==='not' || $behav ===false){
                $option = '_mcmp_ppu_general';
                $behav = $this->get_option_override($option, $product_id);
                if($behav !=='not' && $behav !==false){
                    return true;
                }
            }else{
                return true;
            }
        }
        return false;
    }

    /**
     * Render additional recalculated text and wrap it around the original price
     * @param integer $product_id_to_inspect Product which will receive the additional text - supposed to be checked first
     * @param mixed $price_text Recalculated price
     * @param boolean $row_wrapper Should be a recalc row wrapper?
     * @param boolean $text_wrapper should be the text in wrappers?
     * @return string Automatic additional text 
     * @since 1.9.3
     */
    public function render_recalc_text($product_id_to_inspect , $price_text = '', $row_wrapper = false, $text_wrapper = true)
    {
        
        $pre_text = $this->get_option_override('_mcmp_ppu_recalc_text_prefix', $product_id_to_inspect);
        $pre_text = empty($pre_text) ? '' : esc_html_x( $pre_text, 'Recalculated price prefix text', 'woo-price-per-unit-pro' );
        $suf_text = $this->get_option_override('_mcmp_ppu_recalc_text', $product_id_to_inspect);
        $suf_text = (empty($suf_text) || $suf_text==' ') ? '' : $suf_text;
        if($suf_text=='-automatic-'){
            $suf_text = get_option('_mcmp_ppu_recalc_text_automatic_preposition','/');
            $suf_text = _x( $suf_text, 'Preposition for weight unit when using automatic text', 'woo-price-per-unit-pro' );
			$suf_text = !empty($suf_text) ? '<span class="mcmp-recalc-price-suffix-preposition">'.esc_html(str_replace('%',' ',$suf_text)).'</span>':'';
			
            $recalc_per_units = $this->get_option_override('_mcmp_ppu_recalc_per_units', $product_id_to_inspect, 1);
            if($recalc_per_units!=1){
                $suf_text.=$recalc_per_units.' ';
            }
            //Get custom unit name
            $ratio_unit = $this->get_option_override('_mcmp_ppu_custom_unit', $product_id_to_inspect);
			$ratio_unit = apply_filters('mcmp_ppu_translate_wpml_reg_string', $ratio_unit, 'Price per Unit PRO', 'Predefined custom units names' );
            if(empty($ratio_unit)){
                $ratio_unit = $this->get_option_override('_mcmp_ppu_ratio_unit', $product_id_to_inspect);
                if(empty($ratio_unit)){
                    $ratio_unit = get_option('woocommerce_weight_unit');
                }
            }
            if(array_key_exists( $ratio_unit, $this->plural_translations)){
                $plurals = $this->plural_translations[ $ratio_unit ];
                $ratio_unit = esc_html(translate_nooped_plural( $plurals, $recalc_per_units, 'woo-price-per-unit-pro' ));
            }else{
                $ratio_unit=esc_html_x($ratio_unit, 'Custom unit name', 'woo-price-per-unit-pro');
            }
            $suf_text.=$ratio_unit;
        }else{
			$suf_text = apply_filters('mcmp_ppu_translate_wpml_reg_string', $suf_text, 'Price per Unit PRO', 'Predefined Recalculated price suffix text' );
            $suf_text = esc_html_x($suf_text, 'Recalculated price suffix text', 'woo-price-per-unit-pro');
        }
        if(!empty($pre_text) || !empty($suf_text)){
			if($text_wrapper == true){
				$css_classes = apply_filters('mcmp_ppu_element_classes',get_option('_mcmp_ppu_element_classes'));
			}
        }
        if (!empty($pre_text)) {
			$separator='&nbsp;';
			if($text_wrapper == true){
				$pre_text = '<span class="mcmp-recalc-price-prefix '.$css_classes.'">' . $pre_text . '</span>' . $separator;
            }else{
				$pre_text = $pre_text . $separator;
            }
        }
        if (!empty($suf_text)) {
			$separator=get_option('_mcmp_ppu_recalc_text_separate');
			$separator=($separator=='no')?'':'&nbsp;';
            if($text_wrapper == true){
                $suf_text = $separator . '<span class="mcmp-recalc-price-suffix '.$css_classes.'">' . $suf_text . '</span>';
            }else{
                $suf_text = $separator . $suf_text;
            }
        }
        if($row_wrapper == true){
            $pre_text = '</br><span class="mcmp_recalc_price_row">' . $pre_text;
            $suf_text .= '</span>';
        }

        return $pre_text . $price_text . $suf_text;
    }

    /**
     * Calculates price per unit for a product
     * @param integer $product_to_inspect Product which will be recalculated
     * @param boolean $bypass_check bypass product checking - supposed to be checked - $product_to_inspect must be id of the product
     * @return float calculated price
     * @since 2.0.8
     */
    public function calculate_product_price($product_to_inspect, $bypass_check = false)
    {
        if($bypass_check === true){
            $product_id = $product_to_inspect;
        }else{
            $product_id = $this->check_product_id($product_to_inspect, true);
        }
        if (empty($product_id)) {
            return false;
		}
		$product = wc_get_product($product_id);
		if('variation' == $product->get_type() && get_post_meta($product_id, '_mcmp_ppu_var_show_recalc', true) === ''){
            $behav_general = $this->get_option_override('_mcmp_ppu_general', $product_id);
			if($behav_general == 'not' || $behav_general == 'sel_variants' || $behav_general == 'sel_variants_repl'){
				$behav_single = $this->get_option_override('_mcmp_ppu_single_page', $product_id);
				if($behav_single == 'not' || $behav_single == 'sel_variants' || $behav_single == 'sel_variants_repl'){
					return false;
				}
			}
		}
        $recalc_price = 0;
        $weight = $this->get_option_override('_mcmp_ppu_cust_num_of_units', $product_id);
        if (empty($weight)) {
            $weight = $product->get_weight();
        }
        $weight = floatval($weight);
        $price = floatval($product->get_price());
        if (!empty($weight) && !empty($price)) {
            $price = wc_get_price_to_display($product, array('price' => $price));
            $price = $price / $weight;
            $price = $this->price_ratio_calc($price, $product_id);
            $recalc_price = $price;
        }
        return $recalc_price;
    }

    /** Simplified version of wc_price - doesn't return any html tags
     * Format the price with a currency symbol.
     *
     * @param  float $price Raw price.
     * @param  array $args  Array of Arguments to format a price {
     *     @type bool   $ex_tax_label       Adds exclude tax label. 
     *     @type string $currency           Currency code.
     *     @type string $decimal_separator  Decimal separator.
     *     @type string $thousand_separator Thousand separator.
     *     @type string $decimals           Number of decimals.
     *     @type string $price_format       Price format depending on the currency position.
     * }
     * @return string
     */
    public function mcmp_format_plain_price( $price, $args = array() ) {
        $args = apply_filters('wc_price_args', wp_parse_args($args,
                array(
                    'ex_tax_label'       => false,
                    'currency'           => '',
                    'decimal_separator'  => wc_get_price_decimal_separator(),
                    'thousand_separator' => wc_get_price_thousand_separator(),
                    'decimals'           => wc_get_price_decimals(),
                    'price_format'       => get_woocommerce_price_format(),
                )
            )
        );
        $round_prices = get_option('_mcmp_ppu_disable_price_rounding') == 'yes' ? false : true;
        if ($round_prices == false){
            $price = $this->truncate_number($price, $args['decimals']);
        }
        $negative = $price < 0;
        $price             = floatval( $negative ? $price * -1 : $price );
        $price             = number_format( $price, $args['decimals'], $args['decimal_separator'], $args['thousand_separator'] );
    
        if ( apply_filters( 'woocommerce_price_trim_zeros', false ) && $args['decimals'] > 0 ) {
            $price = wc_trim_zeros( $price );
        }
    
        $price = ( $negative ? '-' : '' ) . sprintf( $args['price_format'], get_woocommerce_currency_symbol( $args['currency'] ), $price );
    
        if ( $args['ex_tax_label'] && wc_tax_enabled() ) {
            $price .= ' ' . WC()->countries->ex_tax_or_vat();
        }
        return $price;
    }

    /**
     * Adds columns to product editor
     * @param array $columns - array of product columns
     * @return array
     * @since 2.0.8
     */
    public function mcmp_ppu_custom_product_columns( $columns ) {
        $columns_new = array(
            'mcmp_recalc_price_text' => esc_html__('Saved recalculated price', 'woo-price-per-unit-pro'),
        );
        $columns=array_merge($columns, $columns_new);
        return $columns;
    }
    
    /**
     * Connect Custom product columns with data
     * @param array $columns - array of product columns
     * @return array
     * @since 2.0.8
     */
    public function mcmp_ppu_manage_product_columns( $column, $post_id ) {
        switch( $column ) {
            /* If displaying the 'offer_end' column. */
            case 'mcmp_recalc_price_text' :
                $product_to_inspect = wc_get_product($post_id);
                if($product_to_inspect->get_type() ==='variable'){
                    $children = $product_to_inspect->get_children();
                }else{
                    $children = array($post_id);
                }
                $recalc_price_text='';
                foreach ($children as $value) {
                    $meta = get_post_meta($value,'_mcmp_ppu_price_text',true);
                    $meta = empty($meta) ? '' : $meta;
                    $recalc_price_text .= (empty($recalc_price_text) ? '' : '</br>') . $meta;
                }    
                echo $recalc_price_text;
                break;
            default :
                break;
        }
    }

    /**
     * Render single variation price - called from filter woocommerce_available_variation
     * @param array $args Information about the variant
     * @param object $product Parent product for the variant
     * @param object $variation Variation for which the calculation takes place
     * @since 1.6
     * @return array original array + new info
     */
    public function custom_variant_price($args, $product, $variation)
    {
        if (is_null($variation) || !is_product() || get_option('_mcmp_ppu_var_display_option_recalc') != 'yes'){
            return $args;
        }
		if (empty($args['price_html'])){
			return $args;
		}
		$variation_id = $this->check_product_id($variation, false);
		if( 'yes' == get_option('_mcmp_ppu_var_option_add_gen_suff_texts')) { 
			$add_text = $this->get_option_override('_mcmp_ppu_additional_text', $variation_id);
			if( ! empty( $add_text ) ){
				$add_text = esc_html_x( $add_text, 'General price suffix text', 'woo-price-per-unit-pro' );
				//Inserting the additional text into the Price tag - only if the text ends with span
				$tag_position = strripos ( $args['price_html'] , '</span>' );
				$price_len = strlen( $args['price_html']);
				if( $tag_position && ( $price_len - 7) == $tag_position ){
					$args['price_html'] = substr ( $args['price_html'] , 0 , $price_len -7 );
					$add_text = $add_text . '</span>';
				}
				$css_classes = apply_filters('mcmp_ppu_element_classes',get_option('_mcmp_ppu_element_classes'));
				$separator = get_option('_mcmp_ppu_recalc_text_separate') == 'no' ? '' : ' ';
				$add_text = '<span class="mcmp-variation-option-price-suffix '.$css_classes.'">' . $separator . $add_text . '</span>';
				$args['price_html'] .= $add_text; 
			}
		}
        if((empty($this->get_recalc_behaviour($product))) && get_option('_mcmp_ppu_var_display_option_recalc_forced') == 'no'){
            return $args;
		}
        $weight = $this->get_option_override('_mcmp_ppu_cust_num_of_units', $variation_id);
        if (empty($weight)) {
            $weight = $args['weight'];
            if (empty($weight)) {
                $weight = $product->get_weight();
            }
        }
        $weight = floatval($weight);
        if (empty($weight)) {
            return $args;
        }
        $round_prices = get_option('_mcmp_ppu_disable_price_rounding') == 'yes' ? false : true;
		$wc_decimals=wc_get_price_decimals();
		if(has_filter('mcmp_ppu_avail_variation_sale_price_before_tax')){
			$orig_price = $normal_price = $regular_price = floatval($variation->get_price());
			$normal_price = apply_filters('mcmp_ppu_avail_variation_sale_price_before_tax', $normal_price, $variation);
			$price_changed = ($normal_price != $orig_price);
			$normal_price = wc_get_price_to_display( $variation, array('price' => $normal_price ) );
		}else{
			$orig_price = $normal_price = $regular_price = floatval( $args['display_price'] );
			$price_changed = false;
		}
        if (empty($normal_price)){
            return $args;
		}
        $normal_price = $normal_price / $weight;
        $normal_price = $this->price_ratio_calc($normal_price, $variation_id);
        if ($round_prices == false){
            $normal_price = $this->truncate_number($normal_price, $wc_decimals);
        }
		$recalc = $this->get_formatted_price($normal_price);
		if (get_option('_mcmp_ppu_hide_sale_price') != 'yes') {
			// Allow only one price change(sale_price or regular_price) - if sale price was changed, use the original sale price as regular price.
			if(false == $price_changed){
				if(has_filter('mcmp_ppu_avail_variation_regular_price_before_tax')){
					$orig_price = $regular_price = floatval($variation->get_regular_price());
					$regular_price = apply_filters('mcmp_ppu_avail_variation_regular_price_before_tax', $regular_price, $variation);
					$price_changed = ($regular_price != $orig_price);
					$regular_price = wc_get_price_to_display( $variation, array('price' => $regular_price ) );
				}else{
					$regular_price = floatval( $args['display_regular_price'] );
				}
			}else{
				// If we use the original sale price, it needs taxes calculation.
				$regular_price = wc_get_price_to_display( $variation, array('price' => $regular_price ) );
			}
			if ($variation->is_on_sale() || true == $price_changed) {
                $regular_price = $regular_price / $weight;
                $regular_price = $this->price_ratio_calc($regular_price, $variation_id);
                if ($round_prices == false){
                    $regular_price = $this->truncate_number($regular_price, $wc_decimals);
                }
                $recalc = '<del>' . $this->get_formatted_price($regular_price) . '</del>&nbsp;<ins>' . $recalc . '</ins>';
            }
        }
        $recalc = $this->render_recalc_text($variation_id, $recalc,true);
        $args['price_html'] .= $recalc;
        return $args;
    }

    /**
     * Render cart and mini-cart price - called from filter woocommerce_cart_item_price
     * @param string $price_text Original text
     * @param array $product_data Information about the product in the cart
     * @param integer $cart_key Id of the cart
     * @since 1.6
     * @return string recalculated $price + custom string
     */
    public function custom_cart_price($price_text, $product_data, $cart_key)
    {
		global $woocommerce;

        if (is_null($product_data)){
            return $price_text;
        }
        $product_id = (empty($product_data['variation_id'])) ? $product_data['product_id'] : $product_data['variation_id'];
        //get_recalc_behaviour also checks the product ID for validity - no need to check it again
        $behav = $this->get_recalc_behaviour($product_id,'cart');
        if (empty($behav)) {
            return $price_text;
        }
        $product = $product_data['data'];
        $weight = $this->get_option_override('_mcmp_ppu_cust_num_of_units', $product_id);
        if (empty($weight)) {
            $weight = $product->get_weight(); //$product_data[data]->get_weight()
        }
        $weight = floatval($weight);
        $normal_price = floatval($product->get_price());
        if (empty($weight) || empty($normal_price)) {
            return $price_text;
		}
		$round_prices = get_option('_mcmp_ppu_disable_price_rounding') == 'yes' ? false : true;
		//Pre Woo 3.3.0 compatibility - display_prices_including_tax is declared there.
		if (method_exists($woocommerce->cart,'display_prices_including_tax')) {
			$incl_tax = $woocommerce->cart->display_prices_including_tax();
		} else {
			$incl_tax = 'excl' !== $woocommerce->cart->tax_display_cart;
		}
		if ( $incl_tax ) {
			$normal_price = wc_get_price_including_tax( $product, array('price' => $normal_price) );
		} else {
			$normal_price = wc_get_price_excluding_tax( $product, array('price' => $normal_price) );
		}
		//Get value after taxes, as the filter is working after tax.
		$orig_price = $regular_price = $normal_price;
		$normal_price = apply_filters('mcmp_ppu_cart_sale_price_after_tax', $normal_price, $product, $product_data);
		$price_changed = ($normal_price != $orig_price);
        $normal_price = $normal_price / $weight;
        $normal_price = $this->price_ratio_calc($normal_price, $product_id);
        if ($round_prices == false){
			$wc_decimals=wc_get_price_decimals();
            $normal_price = $this->truncate_number($normal_price, $wc_decimals);
        }
		$recalc = $this->get_formatted_price($normal_price);
		if( 'yes' == get_option('_mcmp_ppu_cart_show_sale_price') ){
			// Allow only one price change(sale_price or regular_price) - if sale price was changed, use the original sale price as regular price.
			if(false == $price_changed){
				$regular_price = floatval($product->get_regular_price());
				if ( $incl_tax ) {
					$regular_price = wc_get_price_including_tax( $product, array('price' => $regular_price) );
				} else {
					$regular_price = wc_get_price_excluding_tax( $product, array('price' => $regular_price) );
				}
				//Get value after taxes, as the filter is working after tax.
				$orig_price = $regular_price;
				//Filter the price after tax, YITH Dynamic pricing will return false if their sale was not applied.
				if(has_filter('mcmp_ppu_cart_regular_price_after_tax',array($this->yith_dyn_pr_helper, 'get_yith_cart_regular_price'))){
					$yith_regular_price = apply_filters('mcmp_ppu_cart_regular_price_after_tax', $regular_price, $product, $product_data);
					if( false !== $yith_regular_price ){
						$regular_price = $yith_regular_price;
						$price_changed = true;
					}
				}else{
					$regular_price = apply_filters('mcmp_ppu_cart_regular_price_after_tax', $regular_price, $product, $product_data);
					$price_changed = ($regular_price != $orig_price);
				}
			}
			if ($product->is_on_sale() || true == $price_changed) {
				$regular_price = $regular_price / $weight;
				$regular_price = $this->price_ratio_calc($regular_price, $product_id);
				if ($round_prices == false){
					$regular_price = $this->truncate_number($regular_price, $wc_decimals);
				}
				$recalc = '<del>' . $this->get_formatted_price($regular_price) . '</del>&nbsp;<ins>' . $recalc . '</ins>';
			}
		}
        switch ($behav) {
            case 'replace':
                $recalc = $this->render_recalc_text($product_id, $recalc);
                return $recalc;
            break;
            default:
                $recalc = $this->render_recalc_text($product_id, $recalc, true);
                return $price_text . $recalc;
            break;
        }
    }

    /**
     * Price ratio calculation - product id is not verified, supposedly it should be already verified
     * takes custom ratio and calculate it in - the price should be already divided by weight(cannot be here because of variable products calculation)
     * @since 1.7
     * @param float $price - price which should be recalculated
     * @param integer $product_id_to_inspect - id of the product - supposed to be checked first
     * @return float recalculated price
     */
    public function price_ratio_calc($price, $product_id_to_inspect)
    {
        $cust_ratio = 1;
        $recalc_per_units = $this->get_option_override('_mcmp_ppu_recalc_per_units', $product_id_to_inspect, 1);
        $recalc_per_units = floatval($recalc_per_units);
        $ratio_unit = $this->get_option_override('_mcmp_ppu_ratio_unit', $product_id_to_inspect);
        if (!empty($ratio_unit)) {
            $current_unit = get_option('woocommerce_weight_unit');
            $cust_ratio = wc_get_weight(1, $current_unit, $ratio_unit);
        }
        $price *= $cust_ratio * $recalc_per_units;
        return $price;
    }

    /**
     * Modifies the general price text
     * @param integer $product_id_to_inspect Product for which the text will be altered - supposed to be checked first
     * @param mixed $price_text Recalculated price
     * @return string Altered general price text
     * @since 2.0.0
     */
    public function general_price_manipulation($product_id_to_inspect , $price_text = '')
    {
		$css_classes = apply_filters('mcmp_ppu_element_classes',get_option('_mcmp_ppu_element_classes'));
        $product = wc_get_product($product_id_to_inspect);
        $product_type = $product->get_type();
        $var_prefix_text = '';
        switch ($product_type) {
            case 'simple':
                $hide_sale = get_option('_mcmp_ppu_hide_sale_price') == 'yes' ? true : false;
                if ($hide_sale == true && $product->is_on_sale()) {
                    $price_text = floatval($product->get_price());
                    if (!empty($price_text)) {
                        $price_text = wc_get_price_to_display($product, array('price' => $price_text));
                    }
                    $price_text = $this->get_formatted_price($price_text);
                }
            break;
            case 'variable':
                //hide variable max price?
                $hide_max = get_option('_mcmp_ppu_var_hide_max_price') == 'yes' ? true : false;
                if ($hide_max == true) {
                    //needles to remake the price?
                    $variable_price_min = floatval($product->get_variation_price('min',true));
                    $price_text = $this->get_formatted_price($variable_price_min);
                }
                if ($product->is_on_sale()){
                    $show_sale_price = get_option('_mcmp_ppu_var_show_sale_price') == 'yes' ? true : false;
                    if ($show_sale_price == true){
                        $price_text = '<ins>' . $price_text . '</ins>';
                        $variable_regular_price_min = floatval($product->get_variation_regular_price('min',true));
                        $price_regular = '<del>' . $this->get_formatted_price($variable_regular_price_min);
                        if ($hide_max == false){
                            $variable_regular_price_max = floatval($product->get_variation_regular_price('max',true));
                            if ($variable_regular_price_min !== $variable_regular_price_max){
                                $price_regular .= '–' . $this->get_formatted_price($variable_regular_price_max);
                            }
                        }
                        $price_regular .= '</del>';
                        $price_text = $price_regular . $price_text;
                    }
                }
                //fill prefix text for variables
                $var_prefix_text = get_option('_mcmp_ppu_var_prefix_text');
                $var_prefix_text = esc_html_x( $var_prefix_text, 'Variations - variable price prefix', 'woo-price-per-unit-pro' );
                $var_prefix_text = (empty($var_prefix_text)) ? '' : '<span class="mcmp-variable-price-prefix '.$css_classes.'">' . $var_prefix_text . ' ' . '</span>';
                $price_text = $var_prefix_text . $price_text;
            break;
        }
        $add_text = $this->get_option_override('_mcmp_ppu_additional_text', $product_id_to_inspect);
        $add_text = esc_html_x( $add_text, 'General price suffix text', 'woo-price-per-unit-pro' );
        $separator = get_option('_mcmp_ppu_recalc_text_separate') == 'no' ? '' : ' ';
        $add_text = (empty($add_text)) ? '' : '<span class="mcmp-general-price-suffix '.$css_classes.'">' . $separator . $add_text . '</span>';
        return $price_text . $add_text;
    }

    /**
     * Render the output - called from filter woocommerce_get_price_html
     * @param string $price Original text
     * @param object $instance Product for which to recalculate
     * @since 1.0
     * @return string recalculated $price + custom string
     */
    public function custom_price($price, $instance)
    {
		global $woocommerce, $page;
		if (is_null($instance)){
			global $product;
		}else{
			$product=$instance;
        }
        //Product validity check
        $prod_id = $this->check_product_id($product);
        if($prod_id == false){
            return false;
        }
        $product_type = $product->get_type();
		//Do not recalculate single variation - it's not displayed anywhere
		if ($product_type == 'variation' && doing_filter('woocommerce_get_price_html')) {
			return $price;
		}
        $behav = $this->get_recalc_behaviour($prod_id);
        // Recalculate price
        if (!empty($behav)) {
            //Price recalculation
            $round_prices = get_option('_mcmp_ppu_disable_price_rounding') == 'yes' ? false : true;
            $wc_decimals=wc_get_price_decimals();
            $recalc_price = '';
            switch ($product_type) {
				case 'simple':
				case 'variation':
                    $units = $this->get_option_override('_mcmp_ppu_cust_num_of_units', $prod_id);
                    if (empty($units) && $product->has_weight()) {
                        $units = $product->get_weight();
                    }
					$units = floatval( $units );
                    $orig_price = $normal_price = $regular_price = floatval($product->get_price());
                    if ($units > 0 && !empty($normal_price)) {
						$normal_price = apply_filters('mcmp_ppu_simple_sale_price_before_tax', $normal_price, $product);
						$price_changed = ($normal_price != $orig_price);
                        $normal_price = wc_get_price_to_display($product, array('price' => $normal_price));
                        $normal_price = $normal_price / $units;
                        $normal_price = $this->price_ratio_calc($normal_price, $prod_id);
                        if ($round_prices == false){
                            $normal_price = $this->truncate_number($normal_price, $wc_decimals);
						}
						$recalc_price = $this->get_formatted_price($normal_price);
						if(get_option('_mcmp_ppu_hide_sale_price') != 'yes'){
							// Allow only one price change(sale_price or regular_price) - if sale price was changed, use the original sale price as regular price.
							if(false == $price_changed){
								$orig_price = $regular_price = floatval($product->get_regular_price());
								$regular_price = apply_filters('mcmp_ppu_simple_regular_price_before_tax', $regular_price, $product);
								$price_changed = ($regular_price != $orig_price);
							}
							if ($product->is_on_sale() || true == $price_changed) {
								$regular_price = wc_get_price_to_display($product, array('price' => $regular_price));
								$regular_price = $regular_price / $units;
								$regular_price = $this->price_ratio_calc($regular_price, $prod_id);
								if ($round_prices == false){
									$regular_price = $this->truncate_number($regular_price, $wc_decimals);
								}
								$recalc_price = '<del>' . $this->get_formatted_price($regular_price) . '</del><ins>' . $recalc_price . '</ins>';
							}
						}
                    }
                    break;
                case 'variable':
                    //When getting variants the filter will fire this again - this is to speed up the process
                    remove_filter('woocommerce_get_price_html', array($this, 'custom_price'), $this->hook_priority);
                    $variations = $product->get_available_variations();
                    add_filter('woocommerce_get_price_html', array($this, 'custom_price'), $this->hook_priority, 2);
                    if(($behav == 'sel_variants' || $behav == 'sel_variants_repl') && count($variations)>0){
                        $variations = array_filter($variations,function($arr_variation){
                            $option_val = get_post_meta($arr_variation['variation_id'], '_mcmp_ppu_var_show_recalc', true);
                            return $option_val == 'yes' ? true : false;
                        });
                    }
                    $num_of_variants = count($variations);
                    if ($num_of_variants > 0) {
                        $parent_prod_weight = $product->get_weight();
                        foreach($variations as $value){
                            $var_id = $value['variation_id'];
                            $units = $this->get_option_override('_mcmp_ppu_cust_num_of_units', $var_id);
                            if (empty($units)) {
                                $units=!empty($value['weight']) ? $value['weight'] : $parent_prod_weight;
                            }
							$units = floatval ( $units );
                            if(!empty($units) && !empty($value['display_price'])){
								$variation_price = apply_filters ('mcmp_ppu_variable_display_price_after_tax', $value['display_price'], $var_id );
								$var_prices[] = $variation_price;
                                $var_recalc_prices[]= $variation_price / floatval($units);
                            }
                        }
                        if (isset($var_recalc_prices) && !empty($var_recalc_prices)) {
							$hide_max = get_option('_mcmp_ppu_var_hide_max_price') == 'yes' ? true : false;
							if('yes' == get_option('_mcmp_ppu_var_display_corresp_recalc_prices')){
								array_multisort( $var_prices, $var_recalc_prices );
							}else{
								asort($var_recalc_prices);
							}
                            $variable_price_min = reset($var_recalc_prices);
                            $variable_price_min = $this->price_ratio_calc($variable_price_min, $prod_id);
                            if ($round_prices == true){
                                $variable_price_min = round($variable_price_min,$wc_decimals);
                            } else {
                                $variable_price_min = $this->truncate_number($variable_price_min, $wc_decimals);
                            }
                            $recalc_price = $this->get_formatted_price($variable_price_min);
                            if ($hide_max == false) {
                                $variable_price_max = end($var_recalc_prices);
                                $variable_price_max = $this->price_ratio_calc($variable_price_max, $prod_id);
                                if ($round_prices == true){
                                    $variable_price_max = round($variable_price_max,$wc_decimals);
                                } else {
                                    $variable_price_max = $this->truncate_number($variable_price_max, $wc_decimals);
                                }
                                if ($variable_price_min !== $variable_price_max) {
                                    $recalc_price .= '–' . $this->get_formatted_price($variable_price_max);
                                }
                            }
                        }
                    }
                    break;
            }
        }
        switch ($behav) {
			case 'replace':
			case 'sel_variants_repl':
                //Recalc happened - let's replace
                //otherwise render normal text
                if(!empty($recalc_price)){
                    $price = $this->render_recalc_text($prod_id, $recalc_price);
                    if ($product_type == 'variable'){
                        //The _mcmp_ppu_var_prefix_text needs to be displayed even for the replaced price text
                        $var_prefix_text = get_option('_mcmp_ppu_var_prefix_text');
						$var_prefix_text = esc_html_x( $var_prefix_text, 'Variations - variable price prefix', 'woo-price-per-unit-pro' );
						if(!empty($var_prefix_text)){
							$css_classes = apply_filters('mcmp_ppu_element_classes',get_option('_mcmp_ppu_element_classes'));
							$var_prefix_text = '<span class="mcmp-variable-price-prefix '.$css_classes.'">' . $var_prefix_text . ' ' . '</span>';
						}else{
							$var_prefix_text = '';
						}
                        $price = $var_prefix_text . $price;
                    }
                }else{
                    $price = $this->general_price_manipulation($prod_id, $price);
                }
            break;
            case 'add':
			case 'sel_variants':
				if(!empty($price)){
					$price = $this->general_price_manipulation($prod_id, $price);
				}
                if(!empty($recalc_price)){
                    $price .=  $this->render_recalc_text($prod_id, $recalc_price, true);
                }
            break;
			default:
				if(!empty($price)){
					$price = $this->general_price_manipulation($prod_id, $price);
				}
            break;
        }
        return $price;
    }
} 
// END class mcmp_ppu
if ( in_array( 'woocommerce/woocommerce.php', (array) get_option( 'active_plugins' ), true ) || (is_multisite() && array_key_exists( 'woocommerce/woocommerce.php', (array) get_site_option( 'active_sitewide_plugins' ) ) ) ) {
    // Instantiate the class
    $mcmp_ppu_obj = mcmp_PPU_pro::get_instance();
}